/*
 * Keyman is copyright (C) SIL International. MIT License.
 *
 * Keyman Keyboard Processor API - Internal Action Interfaces
 */

#pragma once

#include <stdint.h>
#include <stdlib.h>
#include <keyman/keyman_core_api_bits.h>
#include <keyman/keyman_core_api_vkeys.h>

#if defined(__cplusplus)
extern "C"
{
#endif

// ----------------------------------------------------------------------------------
// Actions APIs are now available only to the keyboard debugger, IMX, and Core unit
// tests (17.0)
// ----------------------------------------------------------------------------------

/*
```
### Action Items
These provide the results of processing a key event to the Platform layer and
should be processed by the Platform layer to issue commands to the os text
services framework to transform the text store in the Client Application, among
other actions.
```c
*/

typedef struct {
  uint8_t expected_type;     // km_core_backspace_type
  uintptr_t expected_value;  // used mainly in unit tests
} km_core_backspace_item;

enum km_core_backspace_type {
  KM_CORE_BT_UNKNOWN    = 0,  // Used at beginning of context; user-initiated backspace
  KM_CORE_BT_CHAR       = 1,  // Deleting a character prior to insertion point
  KM_CORE_BT_MARKER     = 2,  // Deleting a marker prior to insertion point
  KM_CORE_BT_MAX_TYPE_ID
};

typedef struct {
  uint8_t   type;
  uint8_t   _reserved[sizeof(void*)-sizeof(uint8_t)];
  union {
    uint32_t               marker;          // MARKER type
    km_core_option_item    const * option;  // OPT types
    km_core_usv            character;       // CHAR type
    uint8_t                capsLock;        // CAPSLOCK type, 1 to turn on, 0 to turn off; re name see #9833
    km_core_backspace_item backspace;       // BACKSPACE type
  };
} km_core_action_item;

enum km_core_action_type {
  KM_CORE_IT_END         = 0,  // Marks end of action items list.
  KM_CORE_IT_CHAR        = 1,  // A Unicode character has been generated.
  KM_CORE_IT_MARKER      = 2,  // Correlates to kmn's "deadkey" markers.
  KM_CORE_IT_ALERT       = 3,  // The keyboard has triggered a alert/beep/bell.
  KM_CORE_IT_BACK        = 4,  // Delete the codepoint preceding the insertion point.
  KM_CORE_IT_PERSIST_OPT = 5,  // The indicated option needs to be stored.
  KM_CORE_IT_EMIT_KEYSTROKE = 6,  // Emit the current keystroke to the application
  KM_CORE_IT_INVALIDATE_CONTEXT = 7,
          // The processor requests that the context buffer be cleared;
          // for applications where context is cached, this clears the context;
          // for applications where context is read from the focused text store,
          // the context is just re-read and markers flushed.
  KM_CORE_IT_CAPSLOCK    = 8,  // Enable or disable capsLock
  KM_CORE_IT_MAX_TYPE_ID
};

/*
```
### `km_core_state_action_items`
##### Description:
Get the list of action items generated by the last call to
`km_core_process_event`.
##### Return:
A pointer to a `km_core_action_item` list, of `*num_items` in length. This data
becomes invalid when the state object is destroyed, or after a call to
`km_core_process_event`. Do not modify the contents of this data. The returned
array is terminated with a `KM_CORE_IT_END` entry.
##### Parameters:
- __state__: A pointer to the opaque `km_core_state` object to be queried.
- __num_items__:
A pointer to a result variable: The number of items in the action item list
including the `KM_CORE_IT_END` terminator. May be null if not that
information is required.

```c
*/
KMN_API
km_core_action_item const *
km_core_state_action_items(km_core_state const *state,
                          size_t *num_items);

/*
```
### `km_core_state_queue_action_items`
##### Description:
Queue actions for the current keyboard processor state; normally
used in IMX callbacks called during `km_core_process_event`.
##### Return:
- `KM_CORE_STATUS_OK`: On success.
- `KM_CORE_STATUS_INVALID_ARGUMENT`:
In the event the `state` or `action_items` pointer are null.
##### Parameters:
- __state__:        A pointer to the opaque `km_core_state` object to be queried.
- __action_items__: The action items to be added to the core
                    queue. Must be terminated with a `KM_CORE_IT_END` entry.

```c
*/
KMN_API
km_core_status
km_core_state_queue_action_items(km_core_state *state,
                         km_core_action_item const *action_items);

/*
```
### `km_core_process_queued_actions`
##### Description:
Process the keyboard processors queued actions for the opaque state object.
Updates the state object as appropriate and fills out its action list.
The client can add actions externally via the `km_core_state_queue_action_items` and
then request the processing of the actions with this method.

The state action list will be cleared at the start of this call; options and context in
the state may also be modified.
##### Return status:
- `KM_CORE_STATUS_OK`: On success.
- `KM_CORE_STATUS_NO_MEM`:
In the event memory is unavailable to allocate internal buffers.
- `KM_CORE_STATUS_INVALID_ARGUMENT`:
In the event the `state` pointer is null

##### Parameters:
- __state__: A pointer to the opaque state object.

```c
*/
KMN_API
km_core_status
km_core_process_queued_actions(km_core_state *state);

#if defined(__cplusplus)
} // extern "C"
#endif
